import pytest  # For skipping tests
import doctest
import os
import tempfile
from .. import generic
from .. import ssh


def load_tests(loader, tests, ignore):
    tests.addTests(doctest.DocTestSuite(ssh, optionflags=doctest.ELLIPSIS))
    return tests


# TODO: The following tests are disabled, as they depend on the template
# environment:
# - They must run as user "zope".
# - They count on the existence of the user "secconnect", the home directory
#   of which must be writable by "zope".
@pytest.mark.skip(reason="Needs the special users zope and secconnect")
def test_all():
    """ TODO: Most tests need to be run as user 'zope'!
    These tests must be generic and capable of running anytime, by anyone.
    """
    uid = os.getuid()
    assert uid == 1005, "You need to run as user 'zope' (uid=1005)! \n\
    Your uid is %s. As root try: 'sudo -u zope python ssh.py'" % uid

    print('Test of private/public key pair generation and appending')
    privkey = ssh.generate_key(unixuser='secconnect', bits=1024)
    assert privkey.startswith('-----BEGIN RSA PRIVATE KEY-----'), \
        'Private key generation failed. Key is: %s' % privkey

    # check if pubkey was appended to /home/secconnect/.ssh/authorized_keys
    print('Checking if pubkey was appended to authorized_keys')
    pubkey = ssh.get_pub_key(key=privkey)
    f = open('/home/secconnect/.ssh/authorized_keys', 'r')
    found = False
    for line in f:
        if pubkey.strip() in line:
            found = True
            break
    assert found, (
        'Public key not found in authorized_keys. Pubkey: %s' % pubkey
    )

    print('Test of PuTTY private/public key pair generation and appending')
    privkey = ssh.generate_key(usePutty=True, unixuser='secconnect', bits=1024)
    assert privkey.startswith('PuTTY-User-Key-File-2: ssh-rsa'), \
        'No PuTTY compatible key found. Key is: %s' % privkey

    # check if pubkey was appended to /home/secconnect/.ssh/authorized_keys
    print('Checking if pubkey was appended to authorized_keys')
    pubkey = ssh.get_pub_key(key=privkey, inputmode='putty')
    pubkey = ' '.join(pubkey.split()[0:2])
    f = open('/home/secconnect/.ssh/authorized_keys', 'r')
    found = False
    for line in f:
        if pubkey.strip() in line:
            found = True
            break
    assert found, (
        'Public key not found in authorized_keys. Pubkey: %s' % pubkey
    )

    print('Test of private/public key pair generation and appending with'
          ' additional options')
    authopt = 'permitopen=\"127.0.0.1:8080\",no-agent-forwarding'
    privkey = ssh.generate_key(
        unixuser='secconnect', bits=1024, authopt=authopt)
    assert privkey.startswith('-----BEGIN RSA PRIVATE KEY-----'), \
        'Private key generation failed. Key is: %s' % privkey

    # check if pubkey was appended to /home/secconnect/.ssh/authorized_keys
    print('Checking if pubkey was appended to authorized_keys with authopt')
    pubkey = ssh.get_pub_key(key=privkey)
    f = open('/home/secconnect/.ssh/authorized_keys', 'r')
    found = False
    for line in f:
        if pubkey.strip() in line:
            if authopt in line:
                found = True
                break
    assert found, 'Public key with options not found in authorized_keys.' \
        ' Pubkey: %s - Authopt: %s' % (pubkey, authopt)

    # Try to build an self extracting archive to mimic a very simple installer
    installer = ssh.createWinInstaller(
        files={
            'a': 'aaa',
            'b': 'bbb',
            'c': 'ccc'
        }
    )
    fd, instpath = tempfile.mkstemp()
    with open(instpath, 'w+b') as inst:
        inst.write(installer)
    ret, out = generic.safe_syscall(['/usr/bin/file', '-b', instpath])
    os.remove(instpath)
    assert (out.strip() ==
            'PE32 executable (GUI) Intel 80386, for MS Windows'
            ), 'The installer seems to be no windows executable!'

    print('All tests successful')
